<?php if (!defined('ABSPATH')) {
    exit;
}


class BonsyRecmanWpAdmin
{

    /**
     * Construct / Load Admin
     */
    public function __construct()
    {
        add_action('init', [$this, 'init']);
    }


    /**
     * Initialize admin panel
     *
     * @throws \Exception
     */
    public function init(): void
    {

        # Make sure user is logged in
        if (!is_user_logged_in()) return;

        # Show admin bar & delete cache if requested
        if (current_user_can('edit_others_pages')) {
            add_action('admin_bar_menu', [$this, 'adminTopBarMenu'], 99);
            $this->deleteCache();
        }

        # Bail if not admin panel
        if (!is_admin()) return;

        # Show cache success message if transient is registered
        if ($noticeData = get_transient('bonsy-cache-reset-notice-' . get_current_user_id())) {
            $this->notice($noticeData['type'] ?? 'warning', $noticeData['message'] ?? 'Unable to get result from cache reset for Bonsy RecMan Plugin');
            delete_transient('bonsy-cache-reset-notice-' . get_current_user_id());
        }

        # Activate Plugin Updates
        if (current_user_can('update_plugins')) {
            require_once(recman()->getFilePath('includes/updates.php'));
        }

        if (current_user_can('manage_options')) {
            add_action('admin_enqueue_scripts', [$this, 'adminScripts']);
            add_action('admin_menu', [$this, 'adminMenu']);
            add_action('admin_init', [$this, 'adminSettings']);
            add_action('admin_init', [$this, 'maybeFlushRewriteRules']);
        }


    }


    /**
     * Delete Cache
     *
     * @throws \Exception
     */
    private function deleteCache(): void
    {

        if (isset($_GET['recman_delete_cache']) && check_admin_referer('recman_delete_cache')) {

            $error = null;

            try {
                recman()->bonsy->cacheDelete(true);
            } catch (Throwable $e) {
                $error = $e->getMessage();
            }

            if (recman()->singleJobPageId()) {
                flush_rewrite_rules();
            }

            # Use persistent notice if user cannot access plugin admin
            if (!current_user_can('manage_options')) {
                set_transient('bonsy-cache-reset-notice-' . get_current_user_id(), [
                    'type' => $error ? 'error' : 'success',
                    'message' => $error ?: 'Job post cache has been reset'
                ], 10);
                wp_redirect(admin_url());
                exit;
            }

            $this->notice($error ? 'error' : 'success', $error ?: 'Job post cache has been reset');

        }

    }


    /**
     * Add menu to the top bar in admin
     *
     * @param \WP_Admin_Bar $wp_admin_bar
     */
    public function adminTopBarMenu(WP_Admin_Bar $wp_admin_bar): void
    {

        # Define Admin Bar Title
        $wp_admin_bar->add_menu([
            'parent' => false,
            'id' => 'recman',
            'title' => 'RecMan Jobs'
        ]);

        # Delete RecMan Cache
        $wp_admin_bar->add_menu([
            'parent' => 'recman',
            'id' => 'recman-delete-cache',
            'title' => 'Delete cache',
            'href' => wp_nonce_url(admin_url('admin.php?page=recman_settings&recman_delete_cache=1'), 'recman_delete_cache')
        ]);

        # Menu items below this must have this permission
        if (!current_user_can('manage_options')) {
            return;
        }

        $wp_admin_bar->add_menu([
            'parent' => 'recman',
            'id' => 'recman-settings',
            'title' => 'Settings',
            'href' => admin_url('admin.php?page=recman_settings')
        ]);

    }


    /**
     * Include admin CSS and JS scripts
     *
     * @param $hook_suffix
     */
    public function adminScripts($hook_suffix): void
    {
        if ($hook_suffix !== 'toplevel_page_recman_settings') return;
        wp_enqueue_style('bonsy-recman-admin-style', recman()->getFileUrl('admin/css/bonsy-admin-style.css'));
    }


    /**
     * Show RecMan Jobs in admin panel menu
     */
    public function adminMenu(): void
    {
        add_menu_page('RecMan Settings', 'RecMan Jobs', 'manage_options', 'recman_settings', [
            &$this,
            'adminView'
        ], recman()->getFileUrl('admin/images/recman.svg'), 100);
    }


    /**
     * Show admin notice / warning message
     *
     * @param string $type
     * @param string $message
     */
    public function notice(string $type, string $message): void
    {
        add_action('admin_notices', static function () use ($type, $message) {
            echo "<div class='notice notice-$type is-dismissible'><p>$message</p></div>";
        });
    }


    /**
     * Validate RecMan API Key Permission Notice
     *
     * @param bool $output_success
     */
    private function validateApiKeyPermissionNotice(bool $output_success = false): void
    {

        if (!get_option('bonsy_license')) {
            delete_option('bonsy_permission_check');
            return;
        }

        if (!recman()->bonsy->tokenIsSet()) return;

        try {
            recman()->bonsy->validateRecmanApiKey();
            if ($output_success) {
                $this->notice('success', "Your registered RecMan API key seems to be safe.");
            }
            update_option('bonsy_permission_check', date('m'));
        } catch (BonsyRecmanException $e) {
            $this->permissionWarningNotice($e->getMessage());
            update_option('bonsy_permission_check', $e->getMessage());
        }
    }


    /**
     * Show Warning about RecMan API key permission
     *
     * @param string $message
     */
    private function permissionWarningNotice(string $message): void
    {
        $button_url = wp_nonce_url(admin_url('admin.php?page=recman_settings&view=license&bonsy_validate_permission=1'), 'bonsy_validate_permission');
        $this->notice('warning', rtrim($message, ' .') . ". <a href='$button_url'>Click here to check RecMan API permission again</a>");
    }


    /**
     * Admin Actions
     *
     * @throws \Exception
     * @noinspection NestedPositiveIfStatementsInspection
     */
    public function adminActions(): void
    {

        # Factory Reset
        if (isset($_GET['bonsy_factory_reset']) && check_admin_referer('bonsy_factory_reset')) {
            recman()::factoryReset();
            $this->notice('success', 'Bonsy RecMan WP Plugin has been reset');
        }

        # Validate API Key Permission
        if (isset($_GET['bonsy_validate_permission']) && check_admin_referer('bonsy_validate_permission')) {
            $this->validateApiKeyPermissionNotice(true);
        } else if (get_option('bonsy_license') && $permission_state = get_option('bonsy_permission_check')) {
            if ($permission_state !== date('m')) {
                if (strlen((string)$permission_state) > 5) {
                    $this->permissionWarningNotice((string)$permission_state);
                } else {
                    $this->validateApiKeyPermissionNotice();
                }
            }
        }

        if (isset($_GET['view']) && $_GET['view'] === 'save_api_settings') {
            $this->saveFetchSettings(array_merge($_GET ?? [], $_POST ?? []));
        }

        # Trigger of bonsy to check if fetch is available
        recman()->bonsy->getLicense();

        if ($error = recman()->bonsy->getError()) {
            $this->notice('error', $error);
        }

    }


    /**
     * Show RecMan Jobs Admin Page
     *
     * @throws \Exception
     */
    public function adminView(): void
    {

        $this->adminActions();

        $view = $_GET['view'] ?? null;
        $license = get_option('bonsy_license');
        $demo_mode = get_option('bonsy_demo');

        # Registration Page
        if (!$license && !$demo_mode) {
            $this->view(['welcome', 'register', 'demoMode']);
            return;
        }

        if ($view === 'viewCache') {
            $this->view(['jsonCache']);
        } else if ($view === 'shortcode') {
            $this->view(['shortcode']);
        } else if ($view === 'changelog') {
            $this->view(['changelog']);
        } else if ($view === 'license') {
            $this->view(['license', 'subscription', 'apikey', 'reset']);
        } else if ($demo_mode) {
            $this->view(['demoMode_disable', 'jobPages', 'caching', 'reset']);
        } else if ($license) {
            $this->view(['fetchSettings', 'jobPages', 'caching']);
        }

    }


    /**
     * Output Admin Views
     *
     * @param array $views
     *
     * @throws \Exception
     */
    public function view(array $views): void
    {
        array_unshift($views, 'template_start');
        $views[] = 'template_end';
        foreach ($views as $view) {
            try {
                require_once(recman()->getFilePath("admin/view/$view.php"));
            } catch (Exception $e) {
                echo "Unable to get page content for $view. Please update plugin!";
            }
        }
    }


    /**
     * Add Register Sections
     *
     * @param string $name
     * @param callable $callback
     */
    private function addSection(string $name, callable $callback): void
    {
        $name .= '_section';
        add_settings_section($name, $name, static function () {
        }, 'recman_settings');
        $callback($name);
    }


    /**
     * RecMan Admin settings
     */
    public function adminSettings(): void
    {

        $this->addSection('bonsy_license', function ($section) {
            register_setting($section, 'bonsy_license', [
                'sanitize_callback' => [$this, 'validateRegistration']
            ]);
        });

        $this->addSection('bonsy_recman_api_key', function ($section) {
            register_setting($section, 'bonsy_recman_api_key', [
                'sanitize_callback' => [$this, 'updateRecmanApiKey']
            ]);
        });

        $this->addSection('bonsy_demo', function ($section) {
            register_setting($section, 'bonsy_demo', 'boolean');
        });

        $this->addSection('bonsy_job_page', function ($section) {
            register_setting($section, 'bonsy_show_job_locally', 'boolean');
            register_setting($section, 'bonsy_single_job_page', static function ($value) {
                update_option('bonsy_permalinks_updated', time());
                return (int)$value;
            });
            register_setting($section, 'bonsy_custom_job_path', [
                'sanitize_callback' => [$this, 'validateCustomSlug']
            ]);
            register_setting($section, 'bonsy_expired_redirect', 'integer');
            register_setting($section, 'bonsy_google_api_key', 'string');
            register_setting($section, 'bonsy_gutenberg_support', 'boolean');
        });


        # Flush Permalinks on change
        add_action('update_option_bonsy_show_job_locally', [$this, 'flushPermalinks'], 10, 2);
        add_action('update_option_bonsy_custom_job_path', [$this, 'flushPermalinks'], 10, 2);
        add_action('update_option_bonsy_expired_redirect', [$this, 'flushPermalinks'], 10, 2);

        $this->addSection('bonsy_shortcode', function ($section) {
            register_setting($section, 'bonsy_shortcode_active', 'boolean');
            register_setting($section, 'bonsy_shortcode_html_template', 'string');
            register_setting($section, 'bonsy_shortcode_custom_css', 'string');
        });

    }


    /**
     * Flush permalinks
     */
    public function flushPermalinks(): void
    {
        update_option('bonsy_permalinks_updated', time());
    }


    /**
     * Maybe flush rewrite rules
     */
    public function maybeFlushRewriteRules(): void
    {

        $lastUpdated = get_option('bonsy_permalinks_updated');

        if ($lastUpdated && $lastUpdated > strtotime('-1 hour')) {
            flush_rewrite_rules();
            delete_option('bonsy_permalinks_updated');
        }

    }


    /**
     * Validate Registration
     *
     * @param string|null $token
     *
     * @return string|null
     */
    public function validateRegistration(string $token = null): ?string
    {

        if (empty($token)) {
            add_settings_error('bonsy_license', 'invalid-access', 'RecMan API key cannot be empty');
            return null;
        }

        try {
            return recman()->bonsy->register($token);
        } catch (BonsyRecmanException $e) {
            add_settings_error('bonsy_license', 'invalid-access', $e->getMessage() . ': ' . $token);
            return null;
        }

    }


    /**
     * Sanitize & update RecMan API key
     *
     * @param string|null $token
     *
     * @return bool
     * @throws \Exception
     */
    public function updateRecmanApiKey(string $token = null): bool
    {

        if (empty($token)) {
            add_settings_error('bonsy_recman_api_key', 'invalid-access', 'RecMan API key cannot be empty');
        } else {
            try {
                $result = recman()->bonsy->updateRecmanApiKey($token);
                recman()->bonsy->cacheDelete(true);
                return $result;
            } catch (BonsyRecmanException $e) {
                add_settings_error('bonsy_recman_api_key', 'invalid-access', $e->getMessage() . '. Using API key: ' . $token);
            }
        }
        return false;
    }


    /**
     * Sanitize Custom Slug
     *
     * @param string|null $slug
     *
     * @return string|null
     */
    public function validateCustomSlug(string $slug = null): ?string
    {

        if (empty($slug)) return null;

        $slug = trim(trim($slug), '/');
        $slug = sanitize_title($slug);

        return (!empty($slug)) ? $slug : null;

    }


    /**
     * Save fetch settings
     *
     * @param array $settings
     *
     * @throws \Exception
     */
    public function saveFetchSettings(array $settings): void
    {

        try {

            # Include Department Filters
            if (isset($settings['department'], $settings['bonsy_filter_departments'])) {
                update_option('bonsy_filter_departments', true);
                $settings['departmentIds'] = implode(',', $settings['departmentIds'] ?? []);
            } else {
                update_option('bonsy_filter_departments', false);
                $settings['departmentIds'] = '';
            }

            # Include Corporations Filters
            if (isset($settings['corporation'], $settings['bonsy_filter_corporations'])) {
                update_option('bonsy_filter_corporations', true);
                $settings['corporationIds'] = implode(',', $settings['corporationIds'] ?? []);
            } else {
                update_option('bonsy_filter_corporations', false);
                $settings['corporationIds'] = '';
            }

            $expire_count = $settings['includeExpiredCount'] ?? 0;
            update_option('bonsy_expired_count', $expire_count);

            unset($settings['view'], $settings['bonsy_filter_departments'], $settings['bonsy_filter_corporations']);

            recman()->bonsy->updateSettings($settings);
            recman()->bonsy->cacheDelete(true);
            $this->notice('success', 'Settings has been saved');

        } catch (BonsyRecmanException $e) {
            $this->notice('error', $e->getMessage());
        }

    }


}


new BonsyRecmanWpAdmin();
