<?php
/*
Plugin Name: Bonsy RecMan WP
Plugin URI: https://recman.bonsy.no/
Description: Auto-publish RecMan job posts to WordPress.
Version: 2.7.0
Author: Bonsy
Author URI: http://recman.bonsy.no/
Copyright: Bonsy
*/


if( !class_exists( 'BonsyRecmanWp' ) && !function_exists( 'recman' ) ) :

    class BonsyRecmanWp {

        public const  VERSION = '2.7.0';
        private const CACHE_DIR = WP_CONTENT_DIR . "/cache/bonsy-recman/";

        public BonsyRecmanJobs $bonsy;


        /**
         * Plugin Constructor
         *
         * @throws \BonsyRecmanException
         * @throws \Exception
         */
        public function __construct() {

            # Load Bonsy RecMan Jobs
            require_once($this->getFilePath( 'includes/BonsyRecmanJobs.php' ));
            $this->bonsy = new BonsyRecmanJobs( 'wp-' . self::VERSION );
            $this->setBonsySettings();

            add_action( 'init', [$this, 'job_rewrite_rule'], 10, 0 );
            add_action( 'pre_get_posts', [$this, 'job_permalink_check'], 0 );
            add_action( 'plugins_loaded', [$this, 'upgradePluginVersion'], 11, 0 );

            add_action( 'plugins_loaded', function() {
                require_once($this->getFilePath( 'admin/admin.php' ));

                # External flush of cache
                if( isset( $_GET['flushRecManJobCache'] ) ) {
                    $this->bonsy->cacheDelete( true );
                    die( 'RecMan cache has been deleted' );
                }

            } );

            add_action( 'get_header', function() {
                $this->bonsy->setPageUrl( get_permalink() );
            } );

            $this->includeSeoData();

            # Load WP Template Helper
            add_action( 'wp_loaded', function() {
                if( is_admin() ) return;
                require_once($this->getFilePath( 'includes/template.php' ));
                # Load shortcode
                if( get_option( 'bonsy_shortcode_active' ) ) {
                    require_once($this->getFilePath( 'includes/shortcode.php' ));
                }
                if( get_option( 'bonsy_gutenberg_support' ) ) {
                    require_once($this->getFilePath( 'includes/gutenberg.php' ));
                }
                do_action( 'bonsy_recman_plugin_loaded' );
            } );

            $this->disallowSearchEngines();

        }


        /**
         * @return void
         */
        public function upgradePluginVersion(): void {
            if( !$this->singleJobPageId() ) return;
            if( version_compare( get_option( 'bonsy_version' ), self::VERSION, '<' ) ) {
                update_option( 'bonsy_version', self::VERSION );
                flush_rewrite_rules();
            }
        }


        /**
         * If filters are activated this can lead to crawling engines
         * overloads the site with too many request. Block crawling of
         * job post page.
         *
         * @return void
         */
        public function disallowSearchEngines(): void {
            add_filter( 'robots_txt', static function($output, $public) {
                if( !$public ) return;
                $output .= "Disallow: /*?jobs=*";
                $output .= "Disallow: /*?*jobs=*";
                return $output;
            }, 99, 2 );
        }


        /**
         * Initialize Bonsy Recman Jobs API settings
         *
         * @return void
         * @throws \BonsyRecmanException
         */
        private function setBonsySettings(): void {

            # Set Bonsy RecMan Jobs parameters
            $this->bonsy->setCacheFolder( self::CACHE_DIR );

            if( $timezone = get_option( 'timezone_string' ) ) {
                $this->bonsy->setTimezone( $timezone );
            }

            $this->bonsy->setFilters( $_GET ?? [] );
            $this->bonsy->setDomain( get_site_url() );


            if( $license = get_option( 'bonsy_license' ) ) {
                $this->bonsy->setToken( $license );
            } else {
                $this->bonsy->demo();
            }

            $this->bonsy->registerFetchErrorClosure( $this->fetchErrorClosure(), $this->fetchSuccessClosure() );

            $this->awaitFetchIfRecentlyFailed();

        }


        /**
         * A closure to run on fetch errors.
         *
         * @return \Closure
         */
        private function fetchErrorClosure(): Closure {
            return static function() {
                if( function_exists( 'update_option' ) ) {
                    update_option( 'bonsy_await_fetch', time() + 1200 );
                }
            };
        }


        /**
         * A closure to run on fetch success.
         *
         * @return \Closure
         */
        private function fetchSuccessClosure(): Closure {
            return static function() {
                if( function_exists( 'update_option' ) ) {
                    delete_option( 'bonsy_await_fetch' );
                }
            };
        }


        /**
         * Await fetch if server response was invalid
         *
         *
         * @return void
         */
        private function awaitFetchIfRecentlyFailed(): void {
            if( $awaitTime = (int)get_option( 'bonsy_await_fetch' ) ) {
                if( $awaitTime < time() ) {
                    delete_option( 'bonsy_await_fetch' );
                } else {
                    $this->bonsy->awaitNewFetch( $awaitTime );
                }
            }
        }


        /**
         * Get Single Job Page ID if activated
         *
         * @return int
         */
        public function singleJobPageId(): int {
            if( !get_option( 'bonsy_show_job_locally' ) ) return 0;
            return (int)get_option( 'bonsy_single_job_page' );
        }


        /**
         * Rewrite rule for pretty permalinks
         */
        public function job_rewrite_rule(): void {
            if( $page_id = $this->singleJobPageId() ) {
                $slug = trim( $this->getPermalinkBaseUrl( $page_id, false ), '/' );
                $query = 'index.php?page_id=' . $page_id . '&recman_job_permalink=$matches[1]';
                add_rewrite_tag( '%recman_job_permalink%', '([^&]+)' );
                add_rewrite_rule( '^' . $slug . '/([^/]*)/?', $query, 'top' );
            }
        }


        /**
         * Permalinks check
         */
        public function job_permalink_check(): void {

            global $wp_query;

            $permalink = $wp_query->query_vars['recman_job_permalink'] ?? $_GET['recman_job_permalink'] ?? null;

            if( is_null( $permalink ) ) return;

            if( $this->bonsy->setCurrentJobByPermalink( trim( (string)$permalink, '/' ) ) ) return;

            $redirect = get_option( 'bonsy_expired_redirect' );
            $redirect = ($redirect) ? get_permalink( $redirect ) : get_site_url();

            if( !headers_sent() && wp_redirect( $redirect ) ) exit;

            # throw new RuntimeException( 'Unable to find job. Tried to redirect browser, but header was already sent.' );

        }


        /**
         * Load Job Post SEO for page
         *
         * @throws \Exception
         */
        public function includeSeoData(): void {

            # Set single page SEO data
            add_action( 'get_header', function() {
                if( ($id = $this->singleJobPageId()) && is_page( $id ) ) {
                    require_once($this->getFilePath( 'includes/seo.php' ));
                    (new BonsyRecmanWpSeo())->setPermalink( $this->getPermalinkBaseUrl( $id ) )->load();
                }
            } );

            # Remove oEmbed for job posts as we are not able to load content of post through oEmbed.
            add_action( 'wp', function() {
                if( ($id = $this->singleJobPageId()) && is_page( $id ) ) {
                    remove_action( 'wp_head', 'wp_oembed_add_discovery_links' );
                    remove_action( 'wp_head', 'wp_oembed_add_host_js' );
                }
            } );

        }


        /**
         * Get plugin file path
         *
         * @param string $file
         *
         * @return string
         */
        public function getFilePath(string $file): string {
            $path = plugin_dir_path( __FILE__ ) . ltrim( $file, '/' );
            if( file_exists( $path ) && is_readable( $path ) ) return $path;
            throw new RuntimeException( 'Bonsy RecMan WP Plugin file not found!: ' . $path );
        }


        /**
         * Get plugin file URL
         *
         * @param string $file
         *
         * @return string
         */
        public function getFileUrl(string $file = ''): string {
            $this->getFilePath( $file ); # This will throw error if file not found
            return plugin_dir_url( __FILE__ ) . ltrim( $file, '/' );
        }


        /**
         * Generate Permalink
         *
         * @param int|null $id
         *
         * @return string|null
         */
        public function getPermalink(int $id = null): ?string {
            if( $page_id = $this->singleJobPageId() ) {
                $title = $this->bonsy->get( 'urn', $id );
                if( !$title ) return null;
                $base = $this->getPermalinkBaseUrl( $page_id );
                return (!empty( $base )) ? "$base/$title/" : null;
            }
            return null;
        }


        /**
         * Get base URL for permalinks
         *
         * @param int $page_id
         * @param bool $include_site_url
         *
         * @return string
         */
        private function getPermalinkBaseUrl(int $page_id, bool $include_site_url = true): string {

            $base = get_permalink( $page_id );

            if( $custom = get_option( 'bonsy_custom_job_path' ) ) {
                $base = rtrim( get_site_url(), '/' ) . "/$custom";
            } else if( function_exists( 'wp_get_post_parent_id' ) && $parent = wp_get_post_parent_id( $page_id ) ) {
                $base = get_permalink( $parent );
            }

            if( !$include_site_url ) {
                $base = str_replace( get_site_url(), '', $base );
            }

            return rtrim( $base, '/' );

        }


        /**
         * Echo Admin Icon
         *
         * @param $file
         *
         * @throws \Exception
         */
        public function icon($file): void {
            echo file_get_contents( $this->getFilePath( "admin/images/$file" ) );
        }


        /**
         * Factory reset - Delete all bonsy options (settings)
         */
        public static function factoryReset(): void {

            foreach( [
                         'bonsy_license',
                         'bonsy_demo',
                         'bonsy_permalinks_updated',
                         'bonsy_recman_api_key',
                         'bonsy_show_job_locally',
                         'bonsy_single_job_page',
                         'bonsy_custom_job_path',
                         'bonsy_expired_redirect',
                         'bonsy_permission_check',
                         'bonsy_filter_departments',
                         'bonsy_filter_corporations',
                         'bonsy_version',
                         'bonsy_await_fetch',
                         'bonsy_expired_count'
                     ] as $option ) {
                delete_option( $option );
            }

            flush_rewrite_rules();

        }


        /**
         * Plugin activation
         */
        public static function activate(): void {
            register_uninstall_hook( __FILE__, [__CLASS__, 'uninstall'] );
            flush_rewrite_rules();
        }


        /**
         * Plugin deactivation
         *
         * @throws \Exception
         */
        public static function deactivation(): void {
            self::deleteFolderWithFiles( self::CACHE_DIR );
            flush_rewrite_rules();
        }


        /**
         * Plugin uninstall
         *
         * @throws \Exception
         */
        public static function uninstall(): void {
            if( !defined( 'WP_UNINSTALL_PLUGIN' ) ) {
                throw new RuntimeException( 'WP_UNINSTALL_PLUGIN IS NOT DEFINED!' );
            }
            self::factoryReset();
            self::deactivation();
        }


        /**
         * Delete folders wih files
         *
         * @param string $dir
         */
        private static function deleteFolderWithFiles(string $dir): void {

            $dir = rtrim( $dir, '/' );

            if( function_exists( 'scandir' ) && $dir && is_dir( $dir ) ) {

                $items = scandir( $dir );

                foreach( $items as $item ) {

                    if( $item === '.' || $item === '..' ) continue;

                    $item = $dir . '/' . $item;

                    if( is_file( $item ) ) {
                        @unlink( $item );
                        continue;
                    }

                    if( is_dir( $item ) ) self::deleteFolderWithFiles( $item );

                }

                @rmdir( $dir );

            }

        }


    }


    /*
    |-------------------------------------------------------------------------------
    | Start RecMan Jobs
    |-------------------------------------------------------------------------------
    */

    function recman(): BonsyRecmanWp {

        # globals
        global $recman;

        # initialize
        if( !isset( $recman ) ) {
            $recman = new BonsyRecmanWp();
        }

        return $recman;

    }


    # initialize
    recman();


endif; # End check class_exists

register_activation_hook( __FILE__, ['BonsyRecmanWp', 'activate'] );
register_deactivation_hook( __FILE__, ['BonsyRecmanWp', 'deactivation'] );
