<?php /** @noinspection PhpMissingReturnTypeInspection */


class BonsyRecmanException extends Exception {

}


class BonsyRecmanApiData {

    /** Cache variables */
    public array $jobs = [];
    public array $jobs_expired = [];
    public array $jobs_filtered = [];
    public array $permalinks = [];
    public array $license = [];
    public array $updated = [];
    public array $settings = [];
    public array $departments = [];
    public array $corporations = [];
    public array $valid_fields = [];

    /** Loop data */
    public array $loop = [];
    public array $loop_current = [];
    public ?string $current_object = null;

}


class BonsyRecmanJobs {

    private const DEFAULT_LOOP_ID = 'jobs';
    private const DEFAULT_EXPIRED_ID = 'expired';
    private const PLUGIN_NAME = 'Bonsy Recman Jobs Plugin';
    private const BASE_URL = 'https://recman-api.bonsy.no/';
    private const PLUGIN_VERSION = 'wp-2';
    private const SIMULATE_BROKEN_API = false;

    /** User Settings */
    private string $token = '';
    private string $cache_folder = '';
    private bool $demo_mode = false;
    private array $filters = [];
    private string $pageUrl = '';
    private string $domain = '';
    private ?string $plugin_version;
    private bool $expired_active = false;

    private ?BonsyRecmanApiData $data = null;


    /** Cache variables */
    private ?string $error = null;

    /** Class variables */
    private DateTime $date;
    private DateTimeZone $timezone;

    private int $await_fetch_time = 0;

    private ?Closure $fetchErrorClosure = null;
    private ?Closure $removeFetchErrorClosure = null;

    private array $distinct_filter_count = [];


    /**
     * Class Construction
     *
     * @throws \Exception
     */
    public function __construct(string $plugin_version = null) {
        $this->timezone = new DateTimeZone( 'Europe/Oslo' );
        $this->date = new DateTime( 'now', $this->timezone );
        $this->plugin_version = $plugin_version;
    }


    /*
    |--------------------------------------------------------------------------
    | User Settings
    |--------------------------------------------------------------------------
    */

    /**
     * Change default timezone
     *
     * @param string $timezone
     *
     * @noinspection PhpUnused
     */
    public function setTimezone(string $timezone): void {
        $this->timezone = new DateTimeZone( $timezone );
        $this->date->setTimezone( $this->timezone );
    }


    /**
     * Set Token
     *
     * @param string $token
     *
     * @return $this
     * @noinspection PhpUnused
     */
    public function setToken(string $token): self {
        $this->token = $token;
        return $this;
    }


    /**
     * Check if token is set
     *
     * @return bool
     */
    public function tokenIsSet(): bool {
        return !empty( $this->token );
    }


    /**
     * Set filters
     *
     * @param array $filters
     *
     * @return $this
     * @noinspection PhpUnused
     */
    public function setFilters(array $filters): self {
        $this->filters = $filters;
        return $this;
    }


    public function setDomain(string $domain): void {
        $this->domain = $domain;
    }


    /**
     * Set Page URL
     *
     * @param string $url
     *
     * @noinspection PhpUnused
     */
    public function setPageUrl(string $url): void {
        $this->pageUrl = $url;
    }


    /**
     * Activate demo mode
     *
     * @param bool $use_demo_mode
     *
     * @return \BonsyRecmanJobs
     */
    public function demo(bool $use_demo_mode = true): self {
        $this->demo_mode = $use_demo_mode;
        return $this;
    }


    /**
     * Define await time until fetching new data from server
     *
     * @param int $time
     *
     * @return void
     */
    public function awaitNewFetch(int $time): void {
        $this->await_fetch_time = $time;
    }


    public function registerFetchErrorClosure(Closure $errorClosure, Closure $successClosure): void {
        $this->fetchErrorClosure = $errorClosure;
        $this->removeFetchErrorClosure = $successClosure;
    }



    /*
    |--------------------------------------------------------------------------
    | Load job data
    |--------------------------------------------------------------------------
    */


    /**
     * Load jobs from cache or through API
     *
     * @noinspection PhpUnused
     * @throws \BonsyRecmanException
     */
    private function loadedData(): BonsyRecmanApiData {

        if( $this->data instanceof BonsyRecmanApiData ) return $this->data;

        # Throw error if token is not set
        if( !$this->demo_mode ) $this->token();

        # Get cached date, or fetch, or use the latest cache available
        $jobs = $this->cacheContent() ?? $this->fetchJobs() ?? $this->cacheLastContent();

        # Create an instance of data
        $this->data = new BonsyRecmanApiData();

        # Bail if no jobs are loaded
        if( empty( $jobs ) ) return $this->data;

        # Convert result to array
        try {
            $jobs = json_decode( $jobs, true, 512, JSON_THROW_ON_ERROR );
        } catch( JsonException $e ) {
            $this->error = 'Unable to convert Bonsy Recman data to JSON: ' . $e->getMessage();
            return $this->data;
        }


        $active_jobs = [];
        $expired_jobs = [];

        # Separate expired and active jobs
        foreach( $jobs['result'] ?? [] as $job ) {
            if( isset( $job['isExpired'] ) && $job['isExpired'] ) {
                $expired_jobs[] = $job;
            } else {
                $active_jobs[] = $job;
            }
        }

        # Sort expired jobs, newer endDate first
        uasort( $expired_jobs, static function($a, $b) {
            $aDate = $a['endDate'] ?? '0000-00-00';
            $bDate = $b['endDate'] ?? '0000-00-00';
            return strcmp( $bDate, $aDate );
        } );

        # Set data
        $this->data->jobs = $active_jobs;
        $this->data->jobs_expired = $expired_jobs;
        $this->data->permalinks = $jobs['permalinks'] ?? [];
        $this->data->license = $jobs['license'] ?? [];
        $this->data->updated = $jobs['updated'] ?? [];
        $this->data->settings = $jobs['settings'] ?? [];
        $this->data->departments = $jobs['departments'] ?? [];
        $this->data->corporations = $jobs['corporations'] ?? [];

        if( isset( $jobs['success'], $jobs['message'] ) && $jobs['success'] === false ) {
            $this->error = $jobs['message'];
        }

        # Set valid fields
        foreach( $this->data->jobs as $job ) {
            $this->data->valid_fields = array_unique( array_merge( $this->data->valid_fields, array_keys( $job ) ) );
        }

        # Filter jobs
        $this->filterJobs();

        # Return class
        return $this->data;

    }


    /**
     * Filter Results
     *
     * This will take an array of filters.
     * If the key exists it will exclude jobs without the given value for that field.
     */
    private function filterJobs(): void {

        if( is_null( $this->data ) ) return;

        $this->data->jobs_filtered = $this->data->jobs;

        # Loop each job post
        foreach( $this->data->jobs_filtered as $key => $job ) {

            # Check each query
            foreach( $this->filters as $searchKey => $searchValuesString ) {

                $searchKey = strtr( $searchKey, [
                    'jobSearch' => 'searchData'
                ] );

                # Bail if not a valid field
                if( !in_array( $searchKey, $this->data->valid_fields, true ) ) continue;

                if( empty( trim( $searchValuesString ) ) ) continue;

                # Split search by comma
                $searchValues = explode( ',', strtolower( trim( $searchValuesString ) ) );
                $searchValues = array_map( static fn($v) => urldecode( $v ), $searchValues );

                # Define the current job value based on key
                # Lowercase + remove comma in values from job post field
                $jobValue = $job[$searchKey] ?? '';
                $jobValue = str_replace( ',', '', strtolower( trim( (string)$jobValue ) ) );

                # Check if job post field contains any of the given search query
                foreach( $searchValues as $searchValue ) {

                    if( $searchKey === 'searchData' ) {
                        $freeSearchValueItems = explode( ' ', trim( $searchValue ) );
                        foreach( $freeSearchValueItems as $freeSearchValueItem ) {
                            if( empty( trim( $freeSearchValueItem ) ) ) continue;
                            if( strpos( $jobValue, trim( $freeSearchValueItem ) ) !== false ) {
                                continue 3;
                            }
                        }
                    }


                    if( $searchValue === $jobValue ) {
                        continue 2;
                    }
                }

                # Remove job post as the query doesn't match the job post.
                unset( $this->data->jobs_filtered[$key] );

            }
        }

    }


    public function getError(): ?string {
        return $this->error;
    }


    /*
    |--------------------------------------------------------------------------
    | Caching
    |--------------------------------------------------------------------------
    */


    /**
     * Set Cache Folder
     *
     * @param string $dir
     *
     * @throws \BonsyRecmanException
     */
    public function setCacheFolder(string $dir): void {

        $dir = rtrim( $dir, '/' ) . '/';

        if( !is_dir( $dir ) && !mkdir( $dir, 0744, true ) && !is_dir( $dir ) ) {
            throw new BonsyRecmanException( "Unable to find or create cache folder for " . self::PLUGIN_NAME );
        }

        $this->cache_folder = $dir;

    }


    /**
     * Get cache folder
     *
     * @return string
     * @throws \BonsyRecmanException
     */
    private function cacheFolder(): string {
        if( !$this->cache_folder ) throw new BonsyRecmanException( "No cache folder is defined for " . self::PLUGIN_NAME . " please use method setCacheFolder()" );
        return $this->cache_folder;
    }


    /**
     * Get cache interval
     */
    private function cacheInterval(): int {
        $hour = (int)$this->date->format( 'G' );
        if( $hour >= 7 && $hour <= 17 ) return 15;
        if( $hour >= 18 && $hour <= 21 ) return 30;
        return 59;
    }


    /**
     * Get cache filename
     *
     * @return string
     * @throws \BonsyRecmanException
     */
    public function cacheFile(): string {
        $minutes = (int)$this->date->format( 'i' );
        $interval = (int)floor( ($minutes / $this->cacheInterval()) + 1 );
        return $this->cacheFolder() . $this->date->format( 'Y-m-d_H' ) . "-$interval.json";
    }


    /**
     * Get Cached JSON Data
     *
     * @param string|null $file
     *
     * @return string|null
     * @throws \BonsyRecmanException
     */
    private function cacheContent(string $file = null): ?string {

        # Get the expected cache file
        $file = $file ?? $this->cacheFile();

        # Bail if no cache file is found
        if( !is_file( $file ) || !is_readable( $file ) ) return null;

        # Load Cache
        if( ini_get( 'allow_url_fopen' ) ) {

            if( function_exists( 'file_get_contents' ) ) {
                $data = @file_get_contents( $file );
                return $data ?: null;
            }

            if( function_exists( 'fopen' ) && function_exists( 'stream_get_contents' ) ) {
                $handle = fopen( $file, 'rb' );
                $data = @stream_get_contents( $handle );
                return $data ?: null;
            }

        }

        if( function_exists( 'curl_exec' ) ) {
            #$path = str_replace( WP_CONTENT_DIR, WP_CONTENT_URL, $path );
            $conn = curl_init( $file );
            curl_setopt( $conn, CURLOPT_SSL_VERIFYPEER, true );
            curl_setopt( $conn, CURLOPT_FRESH_CONNECT, true );
            curl_setopt( $conn, CURLOPT_RETURNTRANSFER, true );
            $data = curl_exec( $conn );
            curl_close( $conn );
            return $data ?: null;
        }

        return null;

    }


    /**
     * Get all cache folder file list
     *
     * @return array
     * @throws \BonsyRecmanException
     */
    private function cacheFolderFileList(): array {
        $list = [];
        $folder = $this->cacheFolder();
        if( $handle = opendir( $folder ) ) {
            while( false !== ($file = readdir( $handle )) ) {
                if( strpos( $file, '.json' ) !== false ) {
                    $list[] = $folder . $file;
                }
            }
            closedir( $handle );
        }
        return $list;
    }


    /**
     * Delete all cache files
     *
     * @throws \BonsyRecmanException
     */
    public function cacheDelete(bool $force_reload = false): void {

        if( $force_reload ) {
            $this->await_fetch_time = 0;
            $this->fetchJobs( false );
            if( $this->error ) {
                $message = ' Cache not deleted!';
                $message .= ' Bonsy RecMan Plugin was unable to get response from server.';
                $message .= ' Plugin will keep previous job post cache until server is available again.';
                $message .= ' Please contact Bonsy!<br><i>';
                $message .= ' Error: ' . $this->error . '</i>';
                throw new RuntimeException( $message );
            }
        }

        foreach( $this->cacheFolderFileList() as $file ) {
            if( is_file( $file ) ) unlink( $file );
        }
        if( $force_reload ) $this->data = null;

    }


    /**
     * Delete the cache folder
     *
     * @noinspection PhpUnused
     * @throws \BonsyRecmanException
     */
    public function cacheDeleteFolder(): void {
        $this->deleteFolderWithFiles( $this->cacheFolder() );
    }


    /** Delete folder and all its files */
    public function deleteFolderWithFiles(string $dir): void {

        $dir = rtrim( $dir, '/' );

        if( function_exists( 'scandir' ) && $dir && is_dir( $dir ) ) {

            $items = scandir( $dir );

            foreach( $items as $item ) {

                if( $item === '.' || $item === '..' ) continue;

                $item = $dir . '/' . $item;

                if( is_file( $item ) ) {
                    unlink( $item );
                    continue;
                }

                if( is_dir( $item ) ) {
                    $this->deleteFolderWithFiles( $item );
                }

            }

            rmdir( $dir );

        }

    }


    /**
     * Get the latest cache file
     *
     * @return string|null
     * @throws \BonsyRecmanException
     */
    private function cacheLastContent(): ?string {
        $list = $this->cacheFolderFileList();
        rsort( $list );
        return $this->cacheContent( current( $list ) );
    }


    /**
     * Save Cache
     *
     * @throws \BonsyRecmanException
     */
    private function cacheSave($data): void {

        # Get the cache filename
        $file = $this->cacheFile();

        # Delete old cache
        $this->cacheDelete();

        # Save cache
        if( function_exists( 'file_put_contents' ) ) {
            file_put_contents( $file, $data );
        }

        # Return if the file now exists
        if( file_exists( $file ) ) return;

        # Else we will try to save using fopen
        $fp = fopen( $file, 'xb' );
        fwrite( $fp, $data );
        fclose( $fp );

    }


    /*
    |--------------------------------------------------------------------------
    | Fetch jobs from Bonsy
    |--------------------------------------------------------------------------
    */

    private function fetchJobs(bool $cache_result = true): ?string {

        if( $this->await_fetch_time > time() ) {
            $time = $this->await_fetch_time - time();
            $this->error = "Bonsy RecMan Plugin has detected error with fetching API data. Plugin will block new attempts for $time seconds. Try to reset cache to check if server is up and running again, else please contact Bonsy!";
            return null;
        }

        try {

            $result = $this->fetch( array_filter( [
                'token' => $this->demo_mode ? null : $this->token()
            ] ), $this->demo_mode ? 'demo' : '' );

            if( $cache_result ) $this->cacheSave( $result );

            if( is_callable( $this->removeFetchErrorClosure ) ) {
                ($this->removeFetchErrorClosure)();
            }

            return $result;

        } catch( Throwable $e ) {

            if( is_callable( $this->fetchErrorClosure ) ) {
                ($this->fetchErrorClosure)();
            }

            $this->error = $e->getMessage();

        }

        return null;

    }


    /**
     * Custom Error Handler
     *
     * @param int $level
     * @param string $message
     * @param string $file
     * @param int $line
     *
     * @throws \ErrorException
     */
    public function handleError(int $level, string $message, string $file = '', int $line = 0): void {
        if( $level && error_reporting() ) {
            throw new ErrorException( $message, 0, $level, $file, $line );
        }
    }


    /**
     * Fetch API Call
     *
     * @param array $parameters
     * @param string $scope
     *
     * @return string|null
     * @throws \BonsyRecmanException
     */
    private function fetch(array $parameters = [], string $scope = ''): ?string {

        $parameters['domain'] = $this->domain;
        $parameters['plugin_version'] = $this->plugin_version ?? self::PLUGIN_VERSION;
        $parameters['php_version'] = PHP_VERSION;

        $uri = self::BASE_URL;
        $uri .= ($scope) ? trim( $scope, '/' ) . '/' : '';
        $uri .= ($parameters) ? '?' . http_build_query( $parameters ) : '';

        $result = null;
        $error = null;
        $timeout = 10;

        $previous_ignore_abort_value = ignore_user_abort( true );
        $previous_time_limit = (int)ini_get( 'max_execution_time' );
        $new_time_limit = $timeout * 5;

        if( $previous_time_limit < $new_time_limit && $previous_time_limit !== 0 ) {
            set_time_limit( $new_time_limit );
        }

        set_error_handler( [$this, 'handleError'], E_ALL );

        try {
            $result = $this->curlFetch( $uri, $timeout ) ?? $this->streamFetch( $uri, $timeout );
        } catch( Throwable $e ) {
            $error = $e->getMessage();
        }

        restore_error_handler();

        if( $previous_time_limit < $new_time_limit && $previous_time_limit !== 0 ) {
            set_time_limit( $previous_time_limit );
        }

        if( !$previous_ignore_abort_value ) {
            ignore_user_abort( false );
        }

        if( empty( $result ) ) {
            $error = $error ?? '';
            throw new BonsyRecmanException( 'RecMan Plugin Unable to fetch data from Bonsy server: ' . $error . '. Please contact Bonsy support.' );
        }

        try {
            $check = json_decode( $result, true, 512, JSON_THROW_ON_ERROR );
        } catch( JsonException $e ) {
            throw new BonsyRecmanException( 'Unable to get a valid response from Bonsy Recman API server.: ' . $e->getMessage() );
        }

        if( !isset( $check['success'] ) ) {
            throw new BonsyRecmanException( 'No success field in the response from server' );
        }

        return $result;

    }


    /**
     * Fetch from API with cURL
     *
     * @param string $uri
     * @param int $timeout
     *
     * @return string|null
     * @throws \BonsyRecmanException
     */
    private function curlFetch(string $uri, int $timeout): ?string {

        if( self::SIMULATE_BROKEN_API ) {
            sleep( $timeout );
            throw new BonsyRecmanException( 'Plugin is simulating broken API' );
        }

        if( !function_exists( 'curl_exec' ) ) return null;

        $ch = curl_init();

        curl_setopt_array( $ch, [
            CURLOPT_URL => $uri,
            CURLOPT_FAILONERROR => false,
            CURLOPT_HEADER => false,
            CURLOPT_TIMEOUT => $timeout,
            CURLOPT_RETURNTRANSFER => true
        ] );

        $result = curl_exec( $ch );

        if( curl_errno( $ch ) ) {
            $error = curl_error( $ch );
            curl_close( $ch );
            throw new BonsyRecmanException( $error );
        }

        curl_close( $ch );

        return $result ?: null;

    }


    /**
     * Fetch from API with stream
     *
     * @param string $uri
     * @param int $timeout
     *
     * @return string|null
     */
    private function streamFetch(string $uri, int $timeout): ?string {

        $context = stream_context_create( [
            'http' => [
                'ignore_errors' => true,
                'timeout' => (float)$timeout
            ]
        ] );

        if( function_exists( 'file_get_contents' ) ) {
            return file_get_contents( $uri, false, $context );
        }

        return stream_get_contents( fopen( $uri, 'rb', false, $context ) );

    }



    /*
    |--------------------------------------------------------------------------
    | Settings & API Keys
    |--------------------------------------------------------------------------
    */


    /**
     * Register account - will return a license (token)
     *
     * @param string $recman_api_key
     *
     * @return string
     * @throws \BonsyRecmanException
     * @noinspection PhpUnused
     */
    public function register(string $recman_api_key): string {

        $result = $this->fetch( [
            'recman_access_token' => $recman_api_key
        ], 'register' );


        try {
            $result = json_decode( $result, false, 512, JSON_THROW_ON_ERROR );
        } catch( JsonException $e ) {
            throw new BonsyRecmanException( 'Unable to convert Bonsy Recman API request. ' . $e->getMessage() );
        }

        if( isset( $result->token ) ) {
            if( empty( $this->token ) ) $this->setToken( $result->token );
            return (string)$result->token;
        }

        throw new BonsyRecmanException( $result->message ?? 'Unable to register account. Unknown error' );

    }


    /**
     * Update Recman Token - Will return true on success
     *
     * @param string $recman_api_key
     *
     * @return bool
     * @throws \BonsyRecmanException
     * @noinspection PhpUnused
     */
    public function updateRecmanApiKey(string $recman_api_key): bool {

        $result = $this->fetch( [
            'token' => $this->token(),
            'recman_access_token' => $recman_api_key
        ], 'updateRecmanKey' );

        try {
            $result = json_decode( $result, false, 512, JSON_THROW_ON_ERROR );
        } catch( JsonException $e ) {
            throw new BonsyRecmanException( 'Unable to convert Bonsy Recman API request. ' . $e->getMessage() );
        }

        if( isset( $result->success ) && $result->success === true ) {
            return true;
        }

        throw new BonsyRecmanException( $result->message ?? 'Unable to update Recman API Key in your account. Unknown error.' );

    }


    /**
     * @param array $settings
     *
     * @return bool
     * @throws \BonsyRecmanException
     * @noinspection PhpUnused
     */
    public function updateSettings(array $settings): bool {

        $settings = array_replace( [
            'token' => $this->token()
        ], $settings );

        $result = $this->fetch( $settings, 'settings/save' );
        try {
            $result = json_decode( $result, false, 512, JSON_THROW_ON_ERROR );
        } catch( JsonException $e ) {
            throw new BonsyRecmanException( 'Unable to convert Bonsy Recman API request. ' . $e->getMessage() );
        }

        if( isset( $result->success ) && $result->success === true ) {
            return true;
        }

        throw new BonsyRecmanException( $result->message ?? 'Unable to update settings in your account. Unknown error.' );

    }


    /**
     * Validate Recman API Key Settings
     *
     * @return bool
     * @throws \BonsyRecmanException
     * @noinspection PhpUnused
     */
    public function validateRecmanApiKey(): bool {

        $result = $this->fetch( ['token' => $this->token()], 'validate' );
        try {
            $result = json_decode( $result, false, 512, JSON_THROW_ON_ERROR );
        } catch( JsonException $e ) {
            throw new BonsyRecmanException( 'Unable to convert Bonsy Recman API request. ' . $e->getMessage() );
        }

        if( isset( $result->success ) && $result->success === true ) {
            return true;
        }

        throw new BonsyRecmanException( $result->message ?? 'Unable to validate API Recman Key. Unknown error.' );

    }


    /*
    |--------------------------------------------------------------------------
    | Helpers
    |--------------------------------------------------------------------------
    */

    /**
     * Get Required Token
     *
     * @return string
     * @throws \BonsyRecmanException
     */
    private function token(): string {
        if( !$this->token ) {
            throw new BonsyRecmanException( 'Token must be set using the method setToken() in ' . self::PLUGIN_NAME );
        }
        return $this->token;
    }


    /**
     * Get job by ID
     *
     * @param int $id
     *
     * @return array|null
     */
    public function getJobById(int $id): ?array {
        try {
            foreach( $this->loadedData()->jobs as $job ) {
                if( isset( $job['jobPostId'] ) && (int)$job['jobPostId'] === $id ) return $job;
            }
            foreach( $this->loadedData()->jobs_expired as $job ) {
                if( isset( $job['jobPostId'] ) && (int)$job['jobPostId'] === $id ) return $job;
            }
        } catch( BonsyRecmanException $e ) {
            $this->error = $e->getMessage();
        }
        return null;
    }


    /**
     * Check if a jobPostId is active
     *
     * @param int $id
     *
     * @return bool
     */
    public function jobPostIdIsActive(int $id): bool {
        try {
            foreach( $this->loadedData()->jobs as $job ) {
                if( isset( $job['jobPostId'] ) && (int)$job['jobPostId'] === $id ) return true;
            }
        } catch( BonsyRecmanException $e ) {
            $this->error = $e->getMessage();
        }
        return false;
    }


    /**
     * Get ID of job by permalink
     *
     * @param string $permalink
     *
     * @return int|null
     */
    private function getJobIdByPermalink(string $permalink): ?int {
        try {
            return array_key_exists( $permalink, $this->loadedData()->permalinks ) ? (int)$this->loadedData()->permalinks[$permalink] : null;
        } catch( BonsyRecmanException $e ) {
            $this->error = $e->getMessage();
            return null;
        }
    }


    public function setCurrentJobByPermalink(string $permalink): bool {
        $job = $this->getJobByPermalink( $permalink );
        if( !$job ) return false;
        try {
            $this->loadedData()->loop_current[self::DEFAULT_LOOP_ID] = $job;
        } catch( BonsyRecmanException $e ) {
            $this->error = $e->getMessage();
            return false;
        }
        return true;
    }


    public function setExampleByFirstAvailableJob(): void {
        try {
            # No need to reload job example
            if( !empty( $this->loadedData()->loop_current[self::DEFAULT_LOOP_ID] ) ) return;

            foreach( $this->loadedData()->jobs as $job ) {
                if( !isset( $job['jobPostId'] ) ) continue;
                $this->loadedData()->loop_current[self::DEFAULT_LOOP_ID] = $job;
                break;
            }

        } catch( BonsyRecmanException $e ) {
            $this->error = $e->getMessage();
        }
    }


    /**
     * Get Job by permalink
     *
     * @param string $permalink
     *
     * @return array|null
     * @noinspection PhpUnused
     */
    public function getJobByPermalink(string $permalink): ?array {
        $id = $this->getJobIdByPermalink( $permalink );
        if( $id ) {
            return $this->getJobById( $id );
        }
        return null;
    }


    /**
     * Get the last external fetch date
     *
     * @return \DateTime
     * @noinspection PhpUnused
     * @throws \Exception
     */
    public function lastFetchDate(): DateTime {
        $date = $this->loadedData()->updated['date'] ?? 'now';
        $date = new DateTime( $date );
        $date->setTimezone( $this->timezone );
        return $date;
    }


    /**
     * Get the expiry date of Bonsy license
     *
     * @return \DateTime
     * @noinspection PhpUnused
     * @throws \Exception
     */
    public function licenseExpiryDate(): DateTime {
        $date = new DateTime( $this->getLicense()['expires'] ?? 'now' );
        $date->setTimezone( $this->timezone );
        return $date;
    }


    /**
     * Get license data
     *
     * @return array
     * @throws \BonsyRecmanException
     */
    public function getLicense(): array {
        return $this->loadedData()->license;
    }


    /**
     * Get stored settings
     *
     * @return array
     * @throws \BonsyRecmanException
     */
    public function getSettings(): array {
        return $this->loadedData()->settings;
    }


    /**
     * Get department list
     *
     * @return array
     * @throws \BonsyRecmanException
     */
    public function getDepartments(): array {
        return $this->loadedData()->departments;
    }


    /**
     * Get corporation list
     *
     * @throws \BonsyRecmanException
     */
    public function getCorporations(): array {
        return $this->loadedData()->corporations;
    }


    /*
    |---------------------------------------------------------------------------
    | Loops
    |---------------------------------------------------------------------------
    */

    /** Define the loop ID */
    private function loopId(bool $is_object = false): ?string {
        try {
            if( $is_object ) return $this->loadedData()->current_object;
            if( $this->expired_active ) return self::DEFAULT_EXPIRED_ID;
            return self::DEFAULT_LOOP_ID;
        } catch( BonsyRecmanException $e ) {
            $this->error = $e->getMessage();
            return null;
        }
    }


    public function reset_loop(): void {
        $this->data = null;
        $this->expired_active = false;
    }


    /**
     * Check if we have a loop or define a new loop
     *
     * @param string|null $object
     * @param array|null $data
     * @param int $count
     * @param int $offset
     * @return bool
     */
    private function have_loop(?string $object, ?array $data, int $count = 0, int $offset = 0): bool {

        # Define new object
        try {

            $this->loadedData()->current_object = $object;

            $id = $this->loopId( (bool)$object );

            # If no loop is defined, we must define one
            if( !isset( $this->loadedData()->loop[$id] ) ) {

                if( $id === self::DEFAULT_EXPIRED_ID ) {
                    $loop_data = array_values( $data ?? $this->loadedData()->jobs_expired );
                } else {
                    $loop_data = array_values( $data ?? $this->loadedData()->jobs_filtered );
                }

                if( $count > 0 ) {
                    $loop_data = array_splice( $loop_data, $offset, $count );
                }

                $this->loadedData()->loop[$id] = $loop_data;

            }

            # If loop has items, return true
            if( !empty( $this->loadedData()->loop[$id] ?? [] ) ) return true;

            # If the requested loop is an object, the object is empty and can be removed.
            if( $object && isset( $this->data->loop[$id] ) ) unset( $this->data->loop[$id] );

        } catch( BonsyRecmanException $e ) {
            $this->error = $e->getMessage();
        }

        return false;

    }


    /** Get the current loop */
    public function the_loop(bool $is_object = false) {

        $id = $this->loopId( $is_object );

        if( is_null( $id ) ) return null;

        # Reset the current
        try {
            $this->loadedData()->loop_current[$id] = null;

            # Shift the next item to current
            if( !empty( $this->loadedData()->loop[$id] ) ) {
                $this->loadedData()->loop_current[$id] = array_shift( $this->loadedData()->loop[$id] );
            }

            return $this->loadedData()->loop_current[$id];

        } catch( BonsyRecmanException $e ) {
            return null;
        }


    }


    /** Get value from current loop */
    public function get(string $field, int $job_id = null, bool $is_object = false) {

        $id = $this->loopId( $is_object );

        if( is_null( $id ) ) return null;

        if( $job_id && $job = $this->getJobById( $job_id ) ) {
            return $job[$field] ?? null;
        }

        try {
            return $this->loadedData()->loop_current[$id][$field] ?? null;
        } catch( BonsyRecmanException $e ) {
            return null;
        }

    }


    /**
     * Get JobPosts
     *
     * @param bool $filtered
     *
     * @return array
     */
    public function getJobs(bool $filtered = true): array {
        try {
            return ($filtered) ? $this->loadedData()->jobs_filtered : $this->loadedData()->jobs;
        } catch( BonsyRecmanException $e ) {
            return [];
        }
    }


    /*
    |--------------------------------------------------------------------------
    | Template
    |--------------------------------------------------------------------------
    */

    /**
     * Get the number of job posts
     *
     * @param bool $filtered
     *
     * @return int
     * @noinspection PhpUnused
     */
    public function countJobs(bool $filtered = true): int {
        return count( $this->getJobs( $filtered ) );
    }


    /**
     * Get total number of positions available
     *
     * @param bool $filtered
     *
     * @return int
     * @noinspection PhpUnused
     */
    public function countPosition(bool $filtered = true): int {
        $count = 0;
        foreach( $this->getJobs( $filtered ) as $job ) {
            if( isset( $job['numberOfPositions'] ) && is_numeric( $job['numberOfPositions'] ) ) {
                $count += (int)$job['numberOfPositions'];
            }
        }
        return $count;
    }


    /**
     * Get distinct values from a field
     *
     * @param string $field
     * @param bool $filtered
     *
     * @return array
     * @noinspection PhpUnused
     */
    public function distinct(string $field, bool $filtered = false): array {

        $list = [];
        $count = [];

        foreach( $this->getJobs( $filtered ) as $jobs ) {

            $distinct_value = $jobs[$field] ?? null;

            if( is_null( $distinct_value ) ) continue;

            $list[] = $distinct_value;

            $id = $jobs['jobPostId'] ?? null;
            if( !$id ) continue;

            $count[$distinct_value][$id] = $id;

        }

        if( !$filtered ) {

            $filtered_jobs = array_map( static function($job) {
                return $job['jobPostId'] ?? null;
            }, $this->getJobs( true ) );

            foreach( $count as $value => $id ) {
                if( !in_array( key( $id ), $filtered_jobs, false ) ) {
                    unset( $count[$value] );
                }
            }
        }

        foreach( $count as $value => $jobs ) {
            $this->distinct_filter_count[$field][$value] = count( $jobs );
        }

        return array_unique( $list );
    }


    /**
     * Return number of job post with the given filter value
     *
     * @param string $field
     * @param string $value
     *
     * @return int
     */
    public function getDistinctFilterCount(string $field, string $value): int {
        return $this->distinct_filter_count[$field][$value] ?? 0;
    }


    /**
     * Check if filters is active for a given field and optionally a given value
     *
     * @param string $field
     * @param string|null $value
     *
     * @return bool
     * @noinspection PhpUnused
     */
    public function hasFilter(string $field, string $value = null): bool {

        $parameter = $this->filters[$field] ?? null;

        if( is_null( $parameter ) ) return false;
        if( is_null( $value ) ) return true;

        $value = str_replace( ',', '', $value );
        $value = strtolower( trim( $value ) );

        $parameter = strtolower( trim( $parameter ) );

        return strpos( $parameter, $value ) !== false;

    }


    /**
     * Check if there are any jobs to loop through
     *
     * @param int $count
     * @param int $offset
     * @param bool $expired_only
     *
     * @return bool
     * @noinspection PhpUnused
     */
    public function have_jobs(int $count, int $offset, bool $expired_only = false): bool {
        if( $expired_only ) $this->expired_active = true;
        return $this->have_loop( null, null, $count, $offset );
    }


    /**
     * Set the next job in loop
     *
     * @return mixed|null
     * @noinspection PhpUnused
     */
    public function the_job() {
        return $this->the_loop();
    }


    /**
     * Check if there are any object to loop through
     *
     * @param string $object
     *
     * @return bool
     * @noinspection PhpUnused
     */
    public function have_object(string $object): bool {

        $data = $this->get( $object );

        if( $data && is_array( $data ) ) {
            return $this->have_loop( $object, array_values( $data ) );
        }

        return false;

    }


    /**
     * Set the next object to loop
     *
     * @return mixed|null
     * @noinspection PhpUnused
     */
    public function the_object() {
        return $this->the_loop( true );
    }


    /**
     * Get data connected to the current object in loop
     *
     * @param string $field
     * @param int|null $id
     *
     * @return mixed|null
     * @noinspection PhpUnused
     */
    public function get_object(string $field, int $id = null) {
        return $this->get( $field, $id, true );
    }


    /**
     * Get URL of page including current filters
     * A helper to build filter URL for the current page
     * This will include current search queries and append these to the page url.
     * If the given key and value exists, it will remove this search
     *
     * @param string $key
     * @param string $value
     *
     * @return string
     * @noinspection PhpUnused
     */
    public function getFilterUrl(string $key, string $value): string {

        $search = array_map( static fn($v) => explode( ',', $v ), $this->filters );

        # Create the parameter and include current value if exists
        $search[$key] = $search[$key] ?? [];

        # Normalize the value
        $value = str_replace( ',', '', strtolower( $value ) );

        if( in_array( $value, $search[$key], true ) ) {
            unset( $search[$key][array_search( $value, $search[$key], true )] );
        } else {
            $search[$key][] = $value;
        }

        # Remove empty filters
        $removeEmpty = static fn($v) => !empty( $v );

        $search[$key] = array_filter( $search[$key], $removeEmpty );
        $search = array_filter( $search, $removeEmpty );

        $search = array_map( static fn($v) => implode( ',', $v ), $search );
        $search = ['jobs' => '1'] + $search;

        # Return page link with search query
        return ($search) ? $this->pageUrl . '?' . http_build_query( $search ) : $this->pageUrl;

    }


}