<?php if( !defined( 'ABSPATH' ) ) exit;


class BonsyRecmanWpUpdate {

    private string $info_url = 'https://recman.bonsy.no/wp/bonsy-recman-info.json';
    private string $cache_key = 'bonsy_recman_update_key';
    private string $plugin_slug = 'bonsay-recman';
    private string $plugin_file = 'bonsay-recman/bonsay-recman.php';
    private string $version;


    /**
     * Construct
     *
     * https://rudrastyh.com/wordpress/self-hosted-plugin-update.html
     *
     */
    public function __construct() {
        $this->version = recman()::VERSION;

        if( !empty( $_GET['force-check'] ) ) {
            delete_site_transient( 'update_plugins' );
            delete_transient( $this->cache_key );
        }

        add_filter( 'plugins_api', [$this, 'modifyPluginDetails'], 20, 3 );
        add_filter( 'pre_set_site_transient_update_plugins', [$this, 'modifyPluginsTransient'], 10, 1 );
        add_action( 'upgrader_process_complete', [$this, 'purge'], 10, 2 );
    }


    /**
     * Get cache or remote file
     *
     * @return object|null
     * @throws \JsonException
     */
    private function request(): ?object {

        # Will return transient or false
        $remote = get_transient( $this->cache_key );

        # Fetch from server if no cache
        $remote = ($remote) ?: $this->fetchInfo();

        # Bail if no remote
        if( is_null( $remote ) ) return null;

        # Get result
        return json_decode( wp_remote_retrieve_body( $remote ), false, 512, JSON_THROW_ON_ERROR );

    }


    /**
     * Fetch plugin info from server
     *
     * @return array|null
     */
    private function fetchInfo(): ?array {

        $remote = wp_remote_get( $this->info_url, [
            'timeout' => 15,
            'headers' => ['Accept' => 'application/json']
        ] );

        # Bail if not success
        if( wp_remote_retrieve_response_code( $remote ) !== 200 ) return null;

        # Bail if no request body
        if( empty( wp_remote_retrieve_body( $remote ) ) ) return null;

        # Set transient ( cache)
        set_transient( $this->cache_key, $remote, 60 * 60 * 12 );

        # Return result
        return $remote;

    }


    /**
     * Plugin Info
     *
     * @param $result
     * @param $action
     * @param $args
     *
     * @return false|object
     * @throws \JsonException
     * @noinspection PhpUnusedParameterInspection
     */
    public function modifyPluginDetails($result, $action, $args) {

        # Do nothing if you're not getting plugin information right now
        if( 'plugin_information' !== $action ) return false;

        # Do nothing if it is not our plugin
        if( $this->plugin_slug !== $args->slug ) return false;

        // Get updates
        $remote = $this->request();

        if( empty( $remote ) ) return false;

        return (object)[
            'name' => $remote->name ?? '',
            'slug' => $remote->slug ?? '',
            'version' => $remote->version ?? recman()::VERSION,
            'tested' => $remote->tested ?? '',
            'requires' => $remote->requires ?? '',
            'author' => $remote->author ?? '',
            'author_profile' => $remote->author_profile ?? '',
            'download_link' => $remote->download_link ?? '',
            'trunk' => $remote->trunk ?? '',
            'requires_php' => $remote->requires_php ?? '7.4',
            'last_updated' => $remote->last_updated ?? '',
            'sections' => [
                'description' => $remote->sections->description ?? '',
                'installation' => $remote->sections->installation ?? '',
                'changelog' => $remote->sections->changelog ?? ''
            ],
            'banners' => [
                'low' => $remote->banners->low = '',
                'high' => $remote->banners->high = ''
            ]
        ];

    }


    /**
     * Push update
     *
     * @param $transient
     *
     * @return mixed
     * @throws \JsonException
     */
    public function modifyPluginsTransient($transient): object {

        # Bail early if no response (error)
        if( !isset( $transient->response ) ) return $transient;

        $remote = $this->request();

        $result = (object)[
            'slug' => $this->plugin_slug,
            'plugin' => $this->plugin_file,
            'new_version' => $this->version,
            'url' => '',
            'package' => '',
            'icons' => [],
            'banners' => [],
            'requires' => get_bloginfo( 'version' ),
            'requires_php' => '7.4'
        ];

        if( $remote && version_compare( $this->version, $remote->version, '<' ) && version_compare( $remote->requires_php, PHP_VERSION, '<=' ) && version_compare( $remote->requires, get_bloginfo( 'version' ), '<=' ) ) {

            foreach( $remote as $key => $value ) {
                if( isset( $result->{$key} ) ) {
                    $result->{$key} = (is_object( $value )) ? (array)$value : $value;
                }
            }

            $result->url = $remote->author_homepage ?? $result->url;
            $result->new_version = $remote->version ?? $result->version;
            $result->package = $remote->download_link ?? $result->download_link;
            $transient->response[$result->plugin] = $result;

        } else {

            $transient->no_update[$result->plugin] = $result;

        }

        return $transient;

    }


    /**
     * After Update
     *
     * Clean the cache when new plugin version is installed
     *
     * @param $upgrader_object
     * @param $options
     *
     * @noinspection PhpUnusedParameterInspection
     */
    public function purge($upgrader_object, $options): void {
        if( $options['action'] === 'update' && $options['type'] === 'plugin' ) {
            delete_transient( $this->cache_key );
        }
    }


}


new BonsyRecmanWpUpdate();
